<?php
/**
 * سكربت إنشاء مجلد الموظف مباشرة على السيرفر
 * 
 * الاستخدام:
 * 1. ارفع هذا الملف على سيرفرك في نفس مجلد admin.html
 * 2. افتح admin.html واملأ البيانات
 * 3. سيتم إنشاء المجلد والملفات مباشرة على السيرفر
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// معالجة طلب OPTIONS للـ CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// التحقق من أن الطلب POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['result' => 'error', 'message' => 'Method not allowed']);
    exit;
}

// قراءة البيانات المرسلة
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    $data = $_POST;
}

// التحقق من البيانات المطلوبة
$requiredFields = ['employeeName', 'folderName', 'scriptUrl', 'governorates'];
foreach ($requiredFields as $field) {
    if (empty($data[$field])) {
        http_response_code(400);
        echo json_encode(['result' => 'error', 'message' => "حقل '$field' مطلوب"]);
        exit;
    }
}

$employeeName = trim($data['employeeName']);
$folderName = trim($data['folderName'] ?? '');
$scriptUrl = trim($data['scriptUrl']);
$governorates = is_array($data['governorates']) ? $data['governorates'] : explode(',', $data['governorates']);

// التحقق من أن اسم المجلد غير فارغ
if (empty($folderName)) {
    http_response_code(400);
    echo json_encode(['result' => 'error', 'message' => 'اسم المجلد مطلوب ولا يمكن أن يكون فارغاً'], JSON_UNESCAPED_UNICODE);
    exit;
}

// تنظيف اسم المجلد (إزالة الأحرف الخطرة فقط)
// نسمح بـ: a-z, A-Z, 0-9, _, - وجميع الأحرف العربية
$folderName = preg_replace('/[^a-zA-Z0-9_\-\p{Arabic}]/u', '', $folderName);
$folderName = trim($folderName);

// التحقق من أن الاسم ليس فارغاً بعد التنظيف
if (empty($folderName)) {
    http_response_code(400);
    echo json_encode(['result' => 'error', 'message' => 'اسم المجلد غير صالح. يرجى استخدام أحرف إنجليزية، أرقام، _ أو أحرف عربية فقط'], JSON_UNESCAPED_UNICODE);
    exit;
}

// التحقق من طول الاسم (يجب أن يكون بين 1 و 255 حرف)
if (mb_strlen($folderName, 'UTF-8') > 255) {
    http_response_code(400);
    echo json_encode(['result' => 'error', 'message' => 'اسم المجلد طويل جداً. الحد الأقصى 255 حرف'], JSON_UNESCAPED_UNICODE);
    exit;
}

// المسار الأساسي (المجلد الحالي)
$basePath = __DIR__;
$employeeFolderPath = $basePath . '/' . $folderName;

// التحقق من وجود المجلد
if (file_exists($employeeFolderPath)) {
    http_response_code(400);
    echo json_encode(['result' => 'error', 'message' => "المجلد '$folderName' موجود بالفعل. يرجى استخدام اسم مختلف."]);
    exit;
}

// إنشاء المجلد
if (!mkdir($employeeFolderPath, 0755, true)) {
    http_response_code(500);
    echo json_encode(['result' => 'error', 'message' => 'فشل في إنشاء المجلد. تحقق من الصلاحيات.']);
    exit;
}

$filesCreated = [];
$errors = [];

// قراءة index.html وتعديله
$sourceIndexPath = $basePath . '/index.html';
if (file_exists($sourceIndexPath)) {
    $indexContent = file_get_contents($sourceIndexPath);
    $changesMade = 0;
    
    // تعديل SCRIPT_URL
    $scriptUrlPattern = '/const\s+SCRIPT_URL\s*=\s*[\'"]([^\'"]+)[\'"]([^;]*);/';
    if (preg_match($scriptUrlPattern, $indexContent)) {
        $indexContent = preg_replace($scriptUrlPattern, "const SCRIPT_URL = '$scriptUrl';", $indexContent);
        $changesMade++;
    }
    
    // تعديل اسم المندوب في value
    $representativePattern = '/(<input[^>]*name=["\']اسم المندوب["\'][^>]*value=["\'])\s*([^"\']+)(["\'][^>]*>)/';
    if (preg_match($representativePattern, $indexContent)) {
        $indexContent = preg_replace($representativePattern, "$1 $employeeName$3", $indexContent);
        $changesMade++;
    }
    
    // تعديل اسم المندوب في label
    $labelPattern = '/(<label[^>]*for=["\']representativeName["\'][^>]*>)\s*([^<]+)(<\/label>)/';
    if (preg_match($labelPattern, $indexContent)) {
        $indexContent = preg_replace($labelPattern, "$1 $employeeName$3", $indexContent);
        $changesMade++;
    }
    
    // تعديل دالة populateGovernorates
    $governoratesJson = json_encode($governorates, JSON_UNESCAPED_UNICODE);
    
    // قراءة iraqData من الملف الأصلي (البحث عن const iraqData = {...})
    $iraqDataStr = '{}';
    $iraqDataStart = strpos($indexContent, 'const iraqData = {');
    if ($iraqDataStart !== false) {
        // البحث عن موقع { الأول
        $braceStart = strpos($indexContent, '{', $iraqDataStart);
        if ($braceStart !== false) {
            $braceCount = 0;
            $foundStart = false;
            $iraqDataEnd = $braceStart;
            
            for ($i = $braceStart; $i < min($braceStart + 50000, strlen($indexContent)); $i++) {
                if ($indexContent[$i] === '{') {
                    $braceCount++;
                    $foundStart = true;
                } else if ($indexContent[$i] === '}') {
                    $braceCount--;
                    if ($foundStart && $braceCount === 0) {
                        $iraqDataEnd = $i;
                        break;
                    }
                }
            }
            
            if ($iraqDataEnd > $braceStart) {
                // استخراج فقط محتوى الكائن {...}
                $iraqDataStr = substr($indexContent, $braceStart, $iraqDataEnd - $braceStart + 1);
            }
        }
    }
    
    $newFunction = "    function populateGovernorates() {
        const govSelect = document.getElementById('governorateSelect');
        if (!govSelect) {
            console.error('⚠️ لم يتم العثور على governorateSelect');
            setTimeout(populateGovernorates, 100);
            return;
        }
        govSelect.innerHTML = '<option value=\"\">اختر المحافظة</option>';
        
        const allowedGovernorates = " . $governoratesJson . ";
        
        if (typeof iraqData === 'undefined' || !iraqData) {
            const iraqDataLocal = " . $iraqDataStr . ";
            window.iraqData = iraqDataLocal;
            console.log('✅ تم تعيين iraqData في النطاق العام لدعم updateDistricts()');
        }
        
        const dataSource = typeof iraqData !== 'undefined' ? iraqData : (typeof window.iraqData !== 'undefined' ? window.iraqData : " . $iraqDataStr . ");
        
        let addedCount = 0;
        allowedGovernorates.forEach(gov => {
            if (dataSource && dataSource[gov]) {
                const option = document.createElement('option');
                option.value = gov;
                option.textContent = gov;
                govSelect.appendChild(option);
                addedCount++;
            }
        });
        
        if (addedCount > 0) {
            console.log('✅ تم ملء ' + addedCount + ' محافظة في القائمة:', allowedGovernorates.join(', '));
        }
    }";
    
    // البحث يدوياً عن الدالة (الطريقة الأكثر دقة للأقواس المتداخلة)
    $funcStart = strpos($indexContent, 'function populateGovernorates()');
    if ($funcStart !== false) {
        $braceCount = 0;
        $foundStart = false;
        $endIndex = $funcStart;
        
        for ($i = $funcStart; $i < min($funcStart + 2000, strlen($indexContent)); $i++) {
            if ($indexContent[$i] === '{') {
                $braceCount++;
                $foundStart = true;
            } else if ($indexContent[$i] === '}') {
                $braceCount--;
                if ($foundStart && $braceCount === 0) {
                    $endIndex = $i;
                    break;
                }
            }
        }
        
        if ($endIndex > $funcStart) {
            $indexContent = substr($indexContent, 0, $funcStart) . 
                           $newFunction . 
                           substr($indexContent, $endIndex + 1);
            $changesMade++;
        }
    }
    
    // حفظ index.html المعدل
    if (file_put_contents($employeeFolderPath . '/index.html', $indexContent)) {
        $filesCreated[] = 'index.html';
    } else {
        $errors[] = 'فشل في حفظ index.html';
    }
}

// نسخ الملفات الأخرى
$filesToCopy = ['style.css', 'test.css', 'Code.gs.js', 'sheets.gs', 'popper.min.js', 'README.md', 'setup_instructions.md'];
foreach ($filesToCopy as $fileName) {
    $sourcePath = $basePath . '/' . $fileName;
    $destPath = $employeeFolderPath . '/' . $fileName;
    
    if (file_exists($sourcePath)) {
        if (copy($sourcePath, $destPath)) {
            $filesCreated[] = $fileName;
        } else {
            $errors[] = "فشل في نسخ $fileName";
        }
    }
}

// نسخ المجلدات
$foldersToCopy = ['css', 'js'];
foreach ($foldersToCopy as $folderNameToCopy) {
    $sourceFolder = $basePath . '/' . $folderNameToCopy;
    $destFolder = $employeeFolderPath . '/' . $folderNameToCopy;
    
    if (is_dir($sourceFolder)) {
        if (copyDirectory($sourceFolder, $destFolder)) {
            $filesCreated[] = "مجلد $folderNameToCopy";
        } else {
            $errors[] = "فشل في نسخ مجلد $folderNameToCopy";
        }
    }
}

// إنشاء ملف التكوين
$config = [
    'employeeName' => $employeeName,
    'scriptUrl' => $scriptUrl,
    'governorates' => $governorates,
    'folderName' => $folderName,
    'createdAt' => date('Y-m-d H:i:s'),
    'serverPath' => $employeeFolderPath
];

if (file_put_contents($employeeFolderPath . '/employee_config.json', json_encode($config, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT))) {
    $filesCreated[] = 'employee_config.json';
}

// دالة نسخ المجلد بشكل متكرر
function copyDirectory($source, $dest) {
    if (!is_dir($dest)) {
        if (!mkdir($dest, 0755, true)) {
            return false;
        }
    }
    
    $dir = opendir($source);
    if (!$dir) {
        return false;
    }
    
    while (($file = readdir($dir)) !== false) {
        if ($file != '.' && $file != '..') {
            $srcFile = $source . DIRECTORY_SEPARATOR . $file;
            $destFile = $dest . DIRECTORY_SEPARATOR . $file;
            
            if (is_dir($srcFile)) {
                if (!copyDirectory($srcFile, $destFile)) {
                    closedir($dir);
                    return false;
                }
            } else {
                if (!copy($srcFile, $destFile)) {
                    closedir($dir);
                    return false;
                }
            }
        }
    }
    
    closedir($dir);
    return true;
}

// إرجاع النتيجة
$response = [
    'result' => 'success',
    'message' => 'تم إنشاء المجلد والملفات بنجاح',
    'folderName' => $folderName,
    'folderPath' => $employeeFolderPath,
    'filesCreated' => $filesCreated,
    'errors' => $errors,
    'config' => $config
];

echo json_encode($response, JSON_UNESCAPED_UNICODE);
