/**
 * سكربت إعداد موظف جديد
 * ينسخ الملفات ويعدل الإعدادات تلقائياً
 * 
 * الاستخدام:
 * node setup-employee.js "اسم الموظف" "رابط Google Sheets" "المحافظة1,المحافظة2,..."
 * 
 * مثال:
 * node setup-employee.js "علي محمد فاضل" "https://script.google.com/macros/s/ABC123/exec" "بغداد,البصرة"
 */

const fs = require('fs');
const path = require('path');

// الحصول على المعاملات من سطر الأوامر
const args = process.argv.slice(2);

if (args.length < 3) {
    console.error('❌ خطأ: يجب توفير جميع المعاملات المطلوبة');
    console.log('\nالاستخدام:');
    console.log('node setup-employee.js "اسم الموظف" "رابط Google Sheets" "المحافظة1,المحافظة2,..."');
    console.log('\nمثال:');
    console.log('node setup-employee.js "علي محمد فاضل" "https://script.google.com/macros/s/ABC123/exec" "بغداد,البصرة"');
    process.exit(1);
}

const employeeName = args[0];
const scriptUrl = args[1];
const governoratesList = args[2].split(',').map(g => g.trim());

// إنشاء اسم المجلد
const folderName = employeeName.replace(/\s+/g, '_').replace(/[^\w\-_]/g, '');

console.log('🚀 بدء إعداد ملفات الموظف...');
console.log(`📁 اسم الموظف: ${employeeName}`);
console.log(`📁 اسم المجلد: ${folderName}`);
console.log(`🔗 رابط السكربت: ${scriptUrl}`);
console.log(`🗺️  المحافظات: ${governoratesList.join(', ')}`);

// الدالة الرئيسية
async function setupEmployee() {
    try {
        // 1. إنشاء المجلد
        const employeeFolder = path.join(__dirname, folderName);
        if (fs.existsSync(employeeFolder)) {
            console.error(`❌ المجلد ${folderName} موجود بالفعل!`);
            process.exit(1);
        }
        
        fs.mkdirSync(employeeFolder, { recursive: true });
        console.log(`✅ تم إنشاء المجلد: ${employeeFolder}`);

        // 2. قائمة الملفات والمجلدات المطلوب نسخها
        const filesToCopy = [
            'index.html',
            'style.css',
            'test.css',
            'Code.gs.js',
            'sheets.gs',
            'popper.min.js',
            'README.md',
            'setup_instructions.md',
            'css',
            'js',
            '43.png',
            '433.jpg',
            'almanara.icon'
        ];

        // 3. نسخ الملفات والمجلدات
        for (const item of filesToCopy) {
            const sourcePath = path.join(__dirname, item);
            const destPath = path.join(employeeFolder, item);

            if (!fs.existsSync(sourcePath)) {
                console.warn(`⚠️  الملف غير موجود: ${item}`);
                continue;
            }

            const stat = fs.statSync(sourcePath);
            
            if (stat.isDirectory()) {
                // نسخ المجلد
                copyDirectory(sourcePath, destPath);
                console.log(`📁 تم نسخ المجلد: ${item}`);
            } else {
                // نسخ الملف
                fs.copyFileSync(sourcePath, destPath);
                console.log(`📄 تم نسخ الملف: ${item}`);
            }
        }

        // 4. تعديل ملف index.html
        const indexHtmlPath = path.join(employeeFolder, 'index.html');
        let indexContent = fs.readFileSync(indexHtmlPath, 'utf8');

        // تعديل SCRIPT_URL
        const scriptUrlPattern = /const\s+SCRIPT_URL\s*=\s*['"]([^'"]+)['"]/;
        if (scriptUrlPattern.test(indexContent)) {
            indexContent = indexContent.replace(scriptUrlPattern, `const SCRIPT_URL = '${scriptUrl}';`);
            console.log('✅ تم تحديث رابط Google Sheets');
        } else {
            console.warn('⚠️  لم يتم العثور على SCRIPT_URL في index.html');
        }

        // تعديل اسم المندوب في radio button
        const representativePattern = /(<input[^>]*name=["']اسم المندوب["'][^>]*value=["'])([^'"]+)(["'])/;
        if (representativePattern.test(indexContent)) {
            indexContent = indexContent.replace(representativePattern, `$1 ${employeeName}$3`);
            console.log('✅ تم تحديث اسم المندوب');
        } else {
            console.warn('⚠️  لم يتم العثور على حقل اسم المندوب');
        }

        // تعديل قائمة المحافظات
        const governoratePattern = /function\s+populateGovernorates\(\)\s*\{[^}]*\}/s;
        const newGovernorateFunction = `function populateGovernorates() {
        const govSelect = document.getElementById('governorateSelect');
        govSelect.innerHTML = '<option value="">اختر المحافظة</option>';
        
        const allowedGovernorates = ${JSON.stringify(governoratesList)};
        
        // قائمة محافظات العراق الكاملة
        const iraqData = {
            "بغداد": ["الرصافة", "الكرخ", "الكاظمية", "الأعظمية", "مدينة الصدر", "المحمودية", "أبو غريب", "الطارمية", "المدائن", "التاجي", "الراشدية", "الاستقلال", "اليوسفية", "اللطيفية", "جسر ديالى"],
            "البصرة": ["البصرة", "الزبير", "القرنة", "المدينة", "الفاو", "شط العرب", "أبي الخصيب", "الدير", "النشوة", "أم قصر", "سفوان"],
            "نينوى": ["الموصل", "تلعفر", "الحمدانية", "سنجار", "مخمور", "الشيخان", "الحضر", "البعاج", "تلكيف", "عقرة", "برطلة", "بعشيقة"],
            "أربيل": ["أربيل", "كويه", "سوران", "شقلاوة", "ميركسور", "خبات", "جومان", "رواندز", "بنصلاوة", "عنكاوا"],
            "كركوك": ["كركوك", "الحويجة", "داقوق", "دبس", "الرياض", "العباسي", "الرشاد", "تازة", "ليلان"],
            "السليمانية": ["السليمانية", "رانية", "دوكان", "حلبجة", "بنجوين", "كلار", "دربندخان", "شاربازير", "جمجمال", "بيارة", "سيروان"],
            "دهوك": ["دهوك", "زاخو", "سميل", "العمادية", "عقرة", "شيخان", "بامرني", "سرسنك"],
            "الأنبار": ["الرمادي", "الفلوجة", "القائم", "هيت", "حديثة", "عنة", "الرطبة", "راوة", "الخالدية", "الحبانية", "عامرية الصمود", "الكرمة", "البغدادي", "العبيدي", "بروانة", "كبيسة"],
            "ديالى": ["بعقوبة", "الخالص", "المقدادية", "خانقين", "بلدروز", "كفري", "جلولاء", "السعدية", "العظيم", "مندلي", "قزانية", "بهرز"],
            "كربلاء": ["كربلاء", "الهندية", "عين التمر", "الحر", "الجدول الغربي", "الخيرات"],
            "النجف": ["النجف", "الكوفة", "المناذرة", "المشخاب", "العباسية", "الحيرة", "القادسية", "الحيدرية", "الشبكة"],
            "واسط": ["الكوت", "النعمانية", "الحي", "العزيزية", "الصويرة", "بدرة", "الموفقية", "جصان", "شيخ سعد", "الأحرار", "الدبوني", "الحفرية", "الزبيدية"],
            "بابل": ["الحلة", "المحاويل", "المسيب", "الهاشمية", "القاسم", "المدحتية", "الاسكندرية", "سدة الهندية", "جبلة", "الكفل", "الشوملي", "أبي غرق"],
            "صلاح الدين": ["تكريت", "سامراء", "بلد", "الدجيل", "بيجي", "الشرقاط", "الدور", "طوزخورماتو", "الضلوعية", "يثرب", "المعتصم", "الاسحاقي", "آمرلي", "سليمان بيك"],
            "القادسية": ["الديوانية", "الشامية", "عفك", "الحمزة", "الدغارة", "الشافعية", "السنية", "الصلاحية", "المهناوية", "غماس", "البدير", "سومر", "السدير", "نفر"],
            "المثنى": ["السماوة", "الرميثة", "الخضر", "الوركاء", "السوير", "المجد", "الهلال", "النجمي", "بصية"],
            "ميسان": ["العمارة", "الميمونة", "قلعة صالح", "علي الغربي", "الكحلاء", "المجر الكبير", "كميت", "المشرح", "السلام", "العدل", "الخير"],
            "ذي قار": ["الناصرية", "الرفاعي", "الشطرة", "سوق الشيوخ", "الجبايش", "قلعة سكر", "الدواية", "الغراف", "النصر", "الفجر", "سيد دخيل", "الإصلاح", "البطحاء", "الفهود", "المنار", "الطار", "العكيكة", "كرمة بني سعيد"]
        };
        
        // إظهار فقط المحافظات المسموحة
        allowedGovernorates.forEach(gov => {
            if (iraqData[gov]) {
                const option = document.createElement('option');
                option.value = gov;
                option.textContent = gov;
                govSelect.appendChild(option);
            }
        });
    }`;

        if (governoratePattern.test(indexContent)) {
            indexContent = indexContent.replace(governoratePattern, newGovernorateFunction);
            console.log('✅ تم تحديث قائمة المحافظات');
        } else {
            console.warn('⚠️  لم يتم العثور على دالة populateGovernorates');
        }

        // حفظ الملف المعدل
        fs.writeFileSync(indexHtmlPath, indexContent, 'utf8');
        console.log('✅ تم حفظ التعديلات في index.html');

        // 5. إنشاء ملف التكوين
        const config = {
            employeeName: employeeName,
            scriptUrl: scriptUrl,
            governorates: governoratesList,
            folderName: folderName,
            createdAt: new Date().toISOString()
        };
        const configPath = path.join(employeeFolder, 'employee_config.json');
        fs.writeFileSync(configPath, JSON.stringify(config, null, 2), 'utf8');
        console.log('✅ تم إنشاء ملف التكوين');

        console.log('\n🎉 تم إعداد ملفات الموظف بنجاح!');
        console.log(`\n📁 المجلد: ${employeeFolder}`);
        console.log(`\n📝 الملفات الجاهزة في: ${employeeFolder}`);
        console.log('\n✅ يمكنك الآن فتح index.html من المجلد الجديد والبدء بالاستخدام!');

    } catch (error) {
        console.error('❌ حدث خطأ:', error.message);
        process.exit(1);
    }
}

// دالة نسخ المجلد بشكل متكرر
function copyDirectory(source, dest) {
    if (!fs.existsSync(dest)) {
        fs.mkdirSync(dest, { recursive: true });
    }

    const entries = fs.readdirSync(source, { withFileTypes: true });

    for (const entry of entries) {
        const sourcePath = path.join(source, entry.name);
        const destPath = path.join(dest, entry.name);

        if (entry.isDirectory()) {
            copyDirectory(sourcePath, destPath);
        } else {
            fs.copyFileSync(sourcePath, destPath);
        }
    }
}

// تشغيل السكربت
setupEmployee();
